/*********************************************************************
 *	
 * DESCRIPTION:
 *	An implementation of the standard P() and V() semaphores, 
 *	using the System V semaphore calls.  The commands in this
 *	module are designed to correspond as closely as possible to
 *	the global semaphore operations in Chimera.
 *	
 *
 * CONTENTS:
 *	sem_t *semCreate(char *name,int initval);
 *		A semaphore is created using 'key' as a unique identifier.
 *		Any other processes which want to use the same semaphore
 *		must use the sam key, but using a call to semattach() instead.
 *		The semaphore is initialized to the value 'initval'.
 *		Thus, for binary semaphores, set initval = 1.
 *		The first process to use this semaphore should call semcreate().
 *		Returns a Semaphore ID, which is to be used in subsequent
 *		P() and V() operations.
 *		Returns -1 if error, and sets errno.
 *
 *	sem_t *semAttach(char *name);
 *		This routine is similar to semcreate(), except that it does not
 *		initialize the value of the semaphore.  The first process
 *		to use the semaphore should call semcreate(), while all
 *		subsequent processes should call semattach().
 *		Returns a Semaphore ID, which is to be used in subsequent
 *		P() and V() operations.
 *		Returns -1 if error, and sets errno.
 *
 *	int semSet(sem_t *sidptr,int val);
 *		Set the value of semaphore to val.
 *
 *	int semVal(sidptr);
 *		Returns the current value of semaphore pointed to by sid.
 *
 *	int semDestroy(sem_t *sidptr);
 *		Remove semaphore with id 'sid'.  This ensures proper cleanup
 *		is done, otherwise the semaphore will remain in the 
 *		memory until the machine is rebooted.  The last process
 *		to finish using the semaphore should to the semdestroy().
 *		In cases where this is not done, the semaphores can be
 *		freed from the command line: see ipcs(1), ipcrm(1).
 *		Note that everything has User and Group permissions, like
 *		with files, so if you can't remove something, check 
 *		the permissions. (Everything in this module gives rw to
 *		everybody, so it should not be a problem.)
 *
 *	int semP(sem_t *sidptr);
 *		Performs the standart P() (wait,down) operation on the semaphore
 *		defined by sid.
 *
 *	int semV(sem_t *sidptr);
 *		Performs the standard V() (signal,up) operation on the 
 *		semaphore defined by sid.
 *	
 */

#ifndef SEM_H
#define SEM_H

typedef int sem_t;

sem_t *semCreate(char *name,int initval);
sem_t *semAttach(char *name);
int semVal(sem_t *sidptr);
int semSet(sem_t *sidptr,int val);
int semDestroy(sem_t *sidptr);
int semDetach(sem_t *sidptr);
int semP(sem_t *sidptr);
int semV(sem_t *sidptr);

#endif SEM_H


